<?php
/**
 * Aldi Weekly Ads Scraper
 * Scrapes data from Aldi weekly ads using the Flipp API
 */

// Error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

class AldiScraper {
    private $userAgent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
    private $postalCode;
    private $merchantId = 2353; // ALDI merchant ID in Flipp
    
    public function __construct($postalCode = '90001') {
        $this->postalCode = $postalCode;
    }
    
    /**
     * Fetch content from URL using cURL
     */
    private function fetchUrl($url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_USERAGENT, $this->userAgent);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/json',
            'Accept-Language: en-US,en;q=0.9',
        ]);
        
        $content = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        if (version_compare(PHP_VERSION, '8.5.0', '<')) {
            curl_close($ch);
        }
        
        if ($content === false) {
            throw new Exception("cURL error: " . $error);
        }
        
        if ($httpCode !== 200) {
            throw new Exception("HTTP error: " . $httpCode);
        }
        
        return $content;
    }
    
    /**
     * Get weekly ad data from Flipp API
     */
    public function getWeeklyAdData() {
        // Flipp API endpoint that returns ALDI weekly ad items
        $apiUrl = "https://backflipp.wishabi.com/flipp/items/search?" . http_build_query([
            'locale' => 'en-US',
            'postal_code' => $this->postalCode,
            'q' => 'aldi'
        ]);
        
        try {
            $response = $this->fetchUrl($apiUrl);
            $data = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("JSON decode error: " . json_last_error_msg());
            }
            
            return $data;
        } catch (Exception $e) {
            throw new Exception("Failed to fetch weekly ad data: " . $e->getMessage());
        }
    }
    
    /**
     * Get flyers data from Flipp API
     */
    public function getFlyersData() {
        // Alternative endpoint to get flyer PDFs/pages
        $apiUrl = "https://cdn-gateflipp.flippback.com/bf/flipp/data?locale=en-US&postal_code={$this->postalCode}&merchant_code=aldi";
        
        try {
            $response = $this->fetchUrl($apiUrl);
            return json_decode($response, true);
        } catch (Exception $e) {
            return null;
        }
    }
    
    /**
     * Format price for display
     */
    public function formatPrice($price) {
        if ($price === null) {
            return 'See store';
        }
        return '$' . number_format($price, 2);
    }
    
    /**
     * Get validity date range
     */
    public function getDateRange($validFrom, $validTo) {
        if (!$validFrom || !$validTo) {
            return '';
        }
        $from = new DateTime($validFrom);
        $to = new DateTime($validTo);
        return $from->format('M j') . ' - ' . $to->format('M j, Y');
    }
}

// Initialize scraper
$postalCode = isset($_GET['postal_code']) ? htmlspecialchars($_GET['postal_code']) : '90001';
$scraper = new AldiScraper($postalCode);

$error = null;
$data = null;
$items = [];
$flyers = [];
$merchants = [];

try {
    $data = $scraper->getWeeklyAdData();
    $items = $data['items'] ?? [];
    $flyers = $data['flyers'] ?? [];
    $merchants = $data['merchants'] ?? [];
} catch (Exception $e) {
    $error = $e->getMessage();
}

// Group items by category
$categories = [];
foreach ($items as $item) {
    $category = $item['_L1'] ?? 'Other';
    if (!isset($categories[$category])) {
        $categories[$category] = [];
    }
    $categories[$category][] = $item;
}

// Sort categories alphabetically
ksort($categories);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ALDI Weekly Ads Scraper</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=DM+Sans:ital,opsz,wght@0,9..40,100..1000;1,9..40,100..1000&display=swap" rel="stylesheet">
    <style>
        :root {
            --aldi-blue: #00005B;
            --aldi-orange: #FF6200;
            --aldi-yellow: #FFCC00;
            --aldi-light: #F5F5F5;
            --card-shadow: 0 4px 20px rgba(0, 0, 91, 0.12);
            --transition: all 0.3s ease;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'DM Sans', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e8ecf4 100%);
            min-height: 100vh;
            color: #1a1a2e;
        }
        
        .header {
            background: var(--aldi-blue);
            color: white;
            padding: 2rem;
            text-align: center;
            position: relative;
            overflow: hidden;
        }
        
        .header::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,204,0,0.15) 0%, transparent 50%);
            animation: pulse 4s ease-in-out infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.5; }
            50% { transform: scale(1.1); opacity: 0.8; }
        }
        
        .header h1 {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            position: relative;
            z-index: 1;
        }
        
        .header h1 span {
            color: var(--aldi-yellow);
        }
        
        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
            position: relative;
            z-index: 1;
        }
        
        .search-bar {
            background: white;
            padding: 1.5rem 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 1rem;
            flex-wrap: wrap;
        }
        
        .search-bar label {
            font-weight: 600;
            color: var(--aldi-blue);
        }
        
        .search-bar input {
            padding: 0.75rem 1.25rem;
            border: 2px solid #e0e0e0;
            border-radius: 50px;
            font-size: 1rem;
            width: 200px;
            transition: var(--transition);
            font-family: inherit;
        }
        
        .search-bar input:focus {
            outline: none;
            border-color: var(--aldi-orange);
            box-shadow: 0 0 0 3px rgba(255, 98, 0, 0.15);
        }
        
        .search-bar button {
            background: var(--aldi-orange);
            color: white;
            border: none;
            padding: 0.75rem 2rem;
            border-radius: 50px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            font-family: inherit;
        }
        
        .search-bar button:hover {
            background: #e55600;
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(255, 98, 0, 0.4);
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 16px;
            box-shadow: var(--card-shadow);
            text-align: center;
            transition: var(--transition);
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(0, 0, 91, 0.18);
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--aldi-blue);
        }
        
        .stat-label {
            color: #666;
            font-size: 0.95rem;
            margin-top: 0.25rem;
        }
        
        .error-box {
            background: linear-gradient(135deg, #ff6b6b 0%, #ee5a5a 100%);
            color: white;
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .error-box::before {
            content: '⚠️';
            font-size: 1.5rem;
        }
        
        .category-section {
            margin-bottom: 3rem;
        }
        
        .category-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--aldi-blue);
            margin-bottom: 1.5rem;
            padding-bottom: 0.75rem;
            border-bottom: 3px solid var(--aldi-orange);
            display: inline-block;
        }
        
        .products-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1.5rem;
        }
        
        .product-card {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: var(--card-shadow);
            transition: var(--transition);
            display: flex;
            flex-direction: column;
        }
        
        .product-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 12px 40px rgba(0, 0, 91, 0.2);
        }
        
        .product-image {
            width: 100%;
            height: 200px;
            object-fit: contain;
            background: #f8f9fa;
            padding: 1rem;
        }
        
        .product-info {
            padding: 1.25rem;
            flex-grow: 1;
            display: flex;
            flex-direction: column;
        }
        
        .product-name {
            font-weight: 600;
            font-size: 1rem;
            color: #1a1a2e;
            margin-bottom: 0.75rem;
            line-height: 1.4;
            flex-grow: 1;
        }
        
        .product-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: auto;
        }
        
        .product-price {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--aldi-orange);
        }
        
        .product-price.original {
            font-size: 0.9rem;
            color: #999;
            text-decoration: line-through;
            margin-left: 0.5rem;
        }
        
        .product-unit {
            font-size: 0.85rem;
            color: #666;
        }
        
        .product-badge {
            display: inline-block;
            background: var(--aldi-yellow);
            color: var(--aldi-blue);
            padding: 0.25rem 0.75rem;
            border-radius: 50px;
            font-size: 0.75rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .product-dates {
            font-size: 0.8rem;
            color: #888;
            margin-top: 0.5rem;
        }
        
        .no-image {
            width: 100%;
            height: 200px;
            background: linear-gradient(135deg, #f0f0f0 0%, #e0e0e0 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: #999;
            font-size: 3rem;
        }
        
        .footer {
            background: var(--aldi-blue);
            color: white;
            padding: 2rem;
            text-align: center;
            margin-top: 3rem;
        }
        
        .footer a {
            color: var(--aldi-yellow);
            text-decoration: none;
        }
        
        .category-tabs {
            display: flex;
            flex-wrap: wrap;
            gap: 0.5rem;
            margin-bottom: 2rem;
            padding: 1rem;
            background: white;
            border-radius: 12px;
            box-shadow: var(--card-shadow);
        }
        
        .category-tab {
            padding: 0.5rem 1rem;
            background: #f0f0f0;
            border: none;
            border-radius: 50px;
            cursor: pointer;
            font-family: inherit;
            font-size: 0.9rem;
            transition: var(--transition);
            color: #333;
        }
        
        .category-tab:hover, .category-tab.active {
            background: var(--aldi-blue);
            color: white;
        }
        
        .category-tab .count {
            display: inline-block;
            background: rgba(255,255,255,0.2);
            padding: 0.1rem 0.5rem;
            border-radius: 50px;
            font-size: 0.75rem;
            margin-left: 0.5rem;
        }
        
        @media (max-width: 768px) {
            .header h1 {
                font-size: 1.75rem;
            }
            
            .container {
                padding: 1rem;
            }
            
            .products-grid {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
                gap: 1rem;
            }
            
            .product-image {
                height: 150px;
            }
            
            .product-info {
                padding: 1rem;
            }
            
            .product-price {
                font-size: 1.25rem;
            }
        }
        
        .loading {
            text-align: center;
            padding: 4rem;
        }
        
        .loading-spinner {
            width: 50px;
            height: 50px;
            border: 4px solid #f0f0f0;
            border-top-color: var(--aldi-orange);
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🛒 ALDI <span>Weekly Ads</span></h1>
        <p>Fresh deals scraped directly from ALDI's weekly circular</p>
    </div>
    
    <div class="search-bar">
        <form method="get" action="">
            <label for="postal_code">📍 ZIP Code:</label>
            <input type="text" id="postal_code" name="postal_code" value="<?= htmlspecialchars($postalCode) ?>" placeholder="Enter ZIP code" pattern="[0-9]{5}" maxlength="5">
            <button type="submit">Search Deals</button>
        </form>
    </div>
    
    <div class="container">
        <?php if ($error): ?>
            <div class="error-box">
                <div>
                    <strong>Error fetching data:</strong><br>
                    <?= htmlspecialchars($error) ?>
                </div>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($items)): ?>
            <!-- Stats -->
            <div class="stats">
                <div class="stat-card">
                    <div class="stat-number"><?= count($items) ?></div>
                    <div class="stat-label">Products on Sale</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= count($categories) ?></div>
                    <div class="stat-label">Categories</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= count($flyers) ?></div>
                    <div class="stat-label">Active Flyers</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= htmlspecialchars($postalCode) ?></div>
                    <div class="stat-label">ZIP Code</div>
                </div>
            </div>
            
            <!-- Category Quick Links -->
            <div class="category-tabs">
                <strong style="padding: 0.5rem 1rem; color: var(--aldi-blue);">Jump to:</strong>
                <?php foreach ($categories as $category => $categoryItems): ?>
                    <a href="#category-<?= urlencode($category) ?>" class="category-tab">
                        <?= htmlspecialchars($category) ?>
                        <span class="count"><?= count($categoryItems) ?></span>
                    </a>
                <?php endforeach; ?>
            </div>
            
            <!-- Products by Category -->
            <?php foreach ($categories as $category => $categoryItems): ?>
                <div class="category-section" id="category-<?= urlencode($category) ?>">
                    <h2 class="category-title"><?= htmlspecialchars($category) ?></h2>
                    <div class="products-grid">
                        <?php foreach ($categoryItems as $item): ?>
                            <div class="product-card">
                                <?php if (!empty($item['clean_image_url'])): ?>
                                    <img src="<?= htmlspecialchars($item['clean_image_url']) ?>" 
                                         alt="<?= htmlspecialchars($item['name'] ?? 'Product') ?>" 
                                         class="product-image"
                                         loading="lazy"
                                         onerror="this.parentElement.innerHTML='<div class=\'no-image\'>🛒</div>'+this.parentElement.innerHTML.slice(this.outerHTML.length)">
                                <?php else: ?>
                                    <div class="no-image">🛒</div>
                                <?php endif; ?>
                                
                                <div class="product-info">
                                    <?php if (!empty($item['pre_price_text'])): ?>
                                        <span class="product-badge"><?= htmlspecialchars($item['pre_price_text']) ?></span>
                                    <?php endif; ?>
                                    
                                    <div class="product-name"><?= htmlspecialchars($item['name'] ?? 'Unknown Product') ?></div>
                                    
                                    <div class="product-meta">
                                        <div>
                                            <span class="product-price"><?= $scraper->formatPrice($item['current_price'] ?? null) ?></span>
                                            <?php if (!empty($item['original_price'])): ?>
                                                <span class="product-price original"><?= $scraper->formatPrice($item['original_price']) ?></span>
                                            <?php endif; ?>
                                            <?php if (!empty($item['post_price_text'])): ?>
                                                <div class="product-unit"><?= htmlspecialchars($item['post_price_text']) ?></div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <?php if (!empty($item['valid_from']) && !empty($item['valid_to'])): ?>
                                        <div class="product-dates">
                                            Valid: <?= $scraper->getDateRange($item['valid_from'], $item['valid_to']) ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endforeach; ?>
            
        <?php elseif (!$error): ?>
            <div class="loading">
                <div class="loading-spinner"></div>
                <p>No products found. Try a different ZIP code.</p>
            </div>
        <?php endif; ?>
    </div>
    
    <div class="footer">
        <p>Data sourced from <a href="https://www.aldi.us/weekly-specials/weekly-ads" target="_blank">ALDI Weekly Ads</a></p>
        <p style="margin-top: 0.5rem; opacity: 0.7;">Prices and availability may vary by location. Last updated: <?= date('F j, Y \a\t g:i A') ?></p>
    </div>
    
    <script>
        // Smooth scroll for category links
        document.querySelectorAll('.category-tab').forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }
            });
        });
        
        // Lazy load images with intersection observer
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        if (img.dataset.src) {
                            img.src = img.dataset.src;
                            img.removeAttribute('data-src');
                        }
                        observer.unobserve(img);
                    }
                });
            });
            
            document.querySelectorAll('img[data-src]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    </script>
</body>
</html>
